<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

// Check if request method is GET
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

try {
    // Validate required fields
    if (!isset($_GET['id'])) {
        echo json_encode(['success' => false, 'message' => 'Missing delivery ID']);
        exit();
    }
    
    $delivery_id = (int)$_GET['id'];
    
    // Get delivery details with related information
    $stmt = $pdo->prepare("
        SELECT 
            d.*,
            c.name as customer_name,
            c.phone as customer_phone,
            c.email as customer_email,
            c.address as customer_address,
            u1.username as assigned_to_name,
            u1.full_name as assigned_to_full_name,
            u2.username as delivered_by_name,
            u2.full_name as delivered_by_full_name,
            q.quote_number as quotation_number,
            q.total_amount as quotation_amount,
            q.notes as quotation_notes
        FROM deliveries d
        LEFT JOIN customers c ON d.customer_id = c.id
        LEFT JOIN users u1 ON d.assigned_to = u1.id
        LEFT JOIN users u2 ON d.delivered_by = u2.id
        LEFT JOIN quotations q ON d.quotation_id = q.id
        WHERE d.id = ?
    ");
    
    $stmt->execute([$delivery_id]);
    $delivery = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$delivery) {
        echo json_encode(['success' => false, 'message' => 'Delivery not found']);
        exit();
    }
    
    // Get quotation items if delivery is from a quotation
    $quotation_items = [];
    if ($delivery['quotation_id']) {
        $stmt = $pdo->prepare("
            SELECT 
                qi.*,
                p.name as product_name,
                p.description as product_description
            FROM quotation_items qi
            JOIN products p ON qi.product_id = p.id
            WHERE qi.quotation_id = ?
            ORDER BY qi.id
        ");
        $stmt->execute([$delivery['quotation_id']]);
        $quotation_items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    echo json_encode([
        'success' => true,
        'delivery' => $delivery,
        'quotation_items' => $quotation_items
    ]);
    
} catch (PDOException $e) {
    error_log("Database error in get_delivery_details.php: " . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => 'Database error occurred. Please try again.'
    ]);
} catch (Exception $e) {
    error_log("Error in get_delivery_details.php: " . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => 'An error occurred. Please try again.'
    ]);
}
?> 